﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using UnityEngine;
using UnityEngine.XR;

//CarsManager is a singleton class responsible for generating and managing all the player cars in the game
public class CarsManager : MonoBehaviour
{
    //Car objects per class
    public List<GameObject> HighEndCars { get; private set; } = new List<GameObject>();
    public List<GameObject> MediumEndCars { get; private set; } = new List<GameObject>();
    public List<GameObject> LowEndCars { get; private set; } = new List<GameObject>();

    //Prefixes and suffixes for names
    private List<string> _LowEndPrefixes = new List<string>();
    private List<string> _LowEndSuffixes = new List<string>();
    private List<string> _MediumEndPrefixes = new List<string>();
    private List<string> _MediumEndSuffixes = new List<string>();
    private List<string> _HighEndPrefixes = new List<string>();
    private List<string> _HighEndSuffixes = new List<string>();

    //Singleton
    private static CarsManager _Instance;

    public static CarsManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<CarsManager>();
            }

            return _Instance;
        }
    }

    /// <summary>
    /// Intializes the CarsManager by reading in all the prefixes and suffixes
    /// </summary>
    /// <returns>Was the initilzation successful?</returns>
    public bool Initialize()
    {
        try
        {
            //Read in all the prefixes and suffixes
            _LowEndPrefixes = Utilities.ReadIxes("Cars/Low-End/Prefixes");
            _LowEndSuffixes = Utilities.ReadIxes("Cars/Low-End/Suffixes");
            _MediumEndPrefixes = Utilities.ReadIxes("Cars/Medium-End/Prefixes");
            _MediumEndSuffixes = Utilities.ReadIxes("Cars/Medium-End/Suffixes");
            _HighEndPrefixes = Utilities.ReadIxes("Cars/High-End/Prefixes");
            _HighEndSuffixes = Utilities.ReadIxes("Cars/High-End/Suffixes");
            return true;
        }

        catch(Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }

    /// <summary>
    /// Generates a car of the specified class type
    /// </summary>
    /// <param name="type">The class type of car to generate</param>
    /// <returns>The generated car game object</returns>
    private GameObject GenerateCar(Constants.CarType type)
    {
        //Let's create our initial objects
        string folder = "";
        List<string> prefixList = new List<string>();
        List<string> suffixList = new List<string>();
        CarsClassConfiguration classConfig;
        CarsClassParts classParts;
        Transform parentTransform;
        AudioGroup engineGroup;
        ColourGroup colourGroup;

        //Check the types to set our correct sources of data based on the type
        if (type == Constants.CarType.LowEnd)
        {
            folder = "Cars/Low-End";
            prefixList = _LowEndPrefixes;
            suffixList = _LowEndSuffixes;
            classConfig = ConfigurationManager.Instance.Cars.LowEnd;
            classParts = CarPartsManager.Instance.LowEndParts;
            parentTransform = GameManager.Instance.GameObjects.FindChild("GeneratedCars").FindChild("LowEnd").transform;
            engineGroup = AudioManager.Instance.AudioGroups["LowEndEnginesGroup"];
            colourGroup = ColoursManager.Instance.ColourGroups["LowEndCars"];
        }
        
        else if (type == Constants.CarType.MediumEnd)
        {
            folder = "Cars/Medium-End";
            prefixList = _MediumEndPrefixes;
            suffixList = _MediumEndSuffixes;
            classConfig = ConfigurationManager.Instance.Cars.MediumEnd;
            classParts = CarPartsManager.Instance.MediumEndParts;
            parentTransform = GameManager.Instance.GameObjects.FindChild("GeneratedCars").FindChild("MediumEnd").transform;
            engineGroup = AudioManager.Instance.AudioGroups["MediumEndEnginesGroup"];
            colourGroup = ColoursManager.Instance.ColourGroups["MediumEndCars"];
        }

        else
        {
            folder = "Cars/High-End";
            prefixList = _HighEndPrefixes;
            suffixList = _HighEndSuffixes;
            classConfig = ConfigurationManager.Instance.Cars.HighEnd;
            classParts = CarPartsManager.Instance.HighEndParts;
            parentTransform = GameManager.Instance.GameObjects.FindChild("GeneratedCars").FindChild("HighEnd").transform;
            engineGroup = AudioManager.Instance.AudioGroups["HighEndEnginesGroup"];
            colourGroup = ColoursManager.Instance.ColourGroups["HighEndCars"];
        }

        //Generate the name
        Randomizer.Regenerate();
        int prefixIndex = Randomizer.RNG.Next(0, prefixList.Count - 1);
        Randomizer.Regenerate();
        int suffixIndex = Randomizer.RNG.Next(0, suffixList.Count - 1);
        string namePrefix = prefixList[prefixIndex];
        string nameSuffix = suffixList[suffixIndex];
        name = namePrefix + " " + nameSuffix;

        //Generate the price
        Randomizer.Regenerate();
        int price = Randomizer.RNG.Next(classConfig.PriceLowerRange, classConfig.PriceUpperRange);
        price = Convert.ToInt32(Math.Round(price / (float)ConfigurationManager.Instance.Cars.PriceRoundFactor) * ConfigurationManager.Instance.Cars.PriceRoundFactor);

        //Generate the colour
        Randomizer.Regenerate();
        int colourIndex = Randomizer.RNG.Next(0, colourGroup.Colours.Count);
        Color colour = colourGroup.Colours.ElementAt(colourIndex).Value.Colour;

        //Load the base body sprite, create our list of parts
        Sprite bodySprite = Resources.Load<Sprite>(folder + "/Body");
        Dictionary<CarPart, Sprite> parts = new Dictionary<CarPart, Sprite>();

        foreach(CarPart part in classParts.Parts)
        {
            //Loop through each part to try to apply it
            bool shouldSpawnPart = !part.Optional;

            if(!shouldSpawnPart)
            {
                //We shouldn't spawn it explicitly, let's do a random chance of spawning it
                Randomizer.Regenerate();
                int val = Randomizer.RNG.Next(0, 2);
                shouldSpawnPart = Convert.ToBoolean(val);
            }

            if(shouldSpawnPart)
            {
                //We're spawning this part, pick a random one from the selection and add it
                Randomizer.Regenerate();
                int partNum = Randomizer.RNG.Next(1, part.Count + 1);
                Sprite partSprite = Resources.Load<Sprite>(folder + "/" + part.Path + "/" + partNum.ToString());

                parts[part] = partSprite;
            }
        }
        
        //Generate all the handling characteristics
        Randomizer.Regenerate();
        float maxSpeed = (float)Randomizer.RNG.NextDouble(classConfig.SpeedLowerRange, classConfig.SpeedUpperRange);

        Randomizer.Regenerate();
        float maxReverseSpeed = (float)Randomizer.RNG.NextDouble(classConfig.ReverseSpeedLowerRange, classConfig.ReverseSpeedUpperRange);

        Randomizer.Regenerate();
        float acceleration = (float)Randomizer.RNG.NextDouble(classConfig.AccelerationLowerRange, classConfig.AccelerationUpperRange);

        Randomizer.Regenerate();
        float handling = (float)Randomizer.RNG.NextDouble(classConfig.HandlingLowerRange, classConfig.HandlingUpperRange);

        Randomizer.Regenerate();
        float driftAmount = (float)Randomizer.RNG.NextDouble(classConfig.DriftingLowerRange, classConfig.DriftingUpperRange);

        //All generated, let's make the object now
        GameObject carObject = Instantiate(GameController.Instance.CarPrefab);
        carObject.name = name;
        carObject.SetActive(false);
        carObject.transform.parent = parentTransform;
        carObject.GetComponent<CarDefinition>().Initialize(namePrefix, nameSuffix, price, colour, bodySprite, parts, maxSpeed, maxReverseSpeed, acceleration, handling, driftAmount, engineGroup.GetRandomFile());
        carObject.transform.position = Constants.SentinelVector3;

        return carObject;   //All done, return it
    }

    /// <summary>
    /// Regenerates a new selection of cars for each class
    /// </summary>
    public void RegenerateCars()
    {
        //Clear and regenerate high end cars
        HighEndCars.Clear();
        for (int highEnd = 0; highEnd < ConfigurationManager.Instance.Cars.GenerateAmount; highEnd++)
        {
            HighEndCars.Add(GenerateCar(Constants.CarType.HighEnd));
        }

        //Clear and regenerate medium end cars
        MediumEndCars.Clear();
        for (int mediumEnd = 0; mediumEnd < ConfigurationManager.Instance.Cars.GenerateAmount; mediumEnd++)
        {
            MediumEndCars.Add(GenerateCar(Constants.CarType.MediumEnd));
        }

        //Clear and regenerate low end cars
        LowEndCars.Clear();
        for (int lowEnd = 0; lowEnd < ConfigurationManager.Instance.Cars.GenerateAmount; lowEnd++)
        {
            LowEndCars.Add(GenerateCar(Constants.CarType.LowEnd));
        }
    }
}
